# Set the root folders for input and output.
# Assumes 'input' and 'output' folders are in the same directory as the script.
$inputFolder = ".\input"
$outputFolder = ".\output"
$tempFolder = "C:\TEMP"
$tempOutputFolder = Join-Path $tempFolder "output"

if (Test-Path $tempFolder) {
    Get-ChildItem -Path $tempFolder -Recurse -Force -ErrorAction SilentlyContinue | Remove-Item -Force -Recurse -ErrorAction SilentlyContinue
}

# Ensure temp folders exist
@($tempFolder, $tempOutputFolder) | ForEach-Object {
    if (!(Test-Path $_)) {
        New-Item -ItemType Directory -Path $_ | Out-Null
    }
}

$absInputPath = (Resolve-Path -Path $inputFolder).Path
$absOutputPath = (Resolve-Path -Path $outputFolder).Path

Get-ChildItem -Path $inputFolder -Filter "*.mkv" -Recurse | ForEach-Object {
    $file = $_
    $destinationPath = $file.FullName.Replace($absInputPath, $absOutputPath)
    if (Test-Path $destinationPath) {
        Write-Host "➡️ Skipping $($file.FullName) because output already exists."
        return
    }

    Write-Host "Processing $($file.FullName)..."

    # Prepare temp paths
    $tempInput = Join-Path $tempFolder $file.Name
    $relativeOutPath = $file.FullName.Replace($absInputPath, "").TrimStart("\")
    $tempOutput = Join-Path $tempOutputFolder $relativeOutPath
    $tempOutputDir = Split-Path $tempOutput -Parent
    if (!(Test-Path $tempOutputDir)) {
        New-Item -ItemType Directory -Path $tempOutputDir -Force | Out-Null
    }

    # Copy to temp input
    Copy-Item $file.FullName -Destination $tempInput -Force

    # FFmpeg local processing
    $ffmpegArgs = @(
        "-i", $tempInput,
        "-map", "0:v",
        "-c:v", "libsvtav1",
        "-preset", "4",
        "-crf", "28",
        "-pix_fmt", "yuv420p10le",
        "-g", "240",
        "-svtav1-params", "film-grain=4:scm=0:tune=3",
        "-map", "0:a?",
        "-c:a", "copy",
        "-map", "0:s?",
        "-c:s", "copy",
        "-map_metadata", "0",
        "-map_chapters", "0",
        $tempOutput
    )

    & ffmpeg @ffmpegArgs

    if ($LASTEXITCODE -ne 0) {
        Write-Host "❗️ Error processing $($file.FullName)"
        Remove-Item $tempInput -Force -ErrorAction SilentlyContinue
        Remove-Item $tempOutput -Force -ErrorAction SilentlyContinue
        Read-Host "Press ENTER to exit"
        exit $LASTEXITCODE
    }

    # Final move to remote output
    $destinationDir = Split-Path $destinationPath -Parent
    if (!(Test-Path $destinationDir)) {
        New-Item -ItemType Directory -Path $destinationDir -Force | Out-Null
    }
    Move-Item $tempOutput -Destination $destinationPath -Force

    Remove-Item $tempInput -Force
    Write-Host "✅ $($file.FullName) processed successfully."
}

Write-Host "🎉 Finished processing all files."
Read-Host "Press ENTER to close the window"
